/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package printing;

import java.awt.Component;
import java.awt.print.PrinterJob;
import java.io.IOException;
import java.net.URISyntaxException;

import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.standard.Copies;
import javax.print.attribute.standard.JobName;

import com.inet.report.EngineRenderData;
import com.inet.viewer.Progress;
import com.inet.viewer.RenderData;
import com.inet.viewer.ReportView;
import com.inet.viewer.SwingReportView;
import com.inet.viewer.SwingReportViewer;

/**
 * This sample show how you print with an existing viewer. For example this can used in your own ViewerContext.
 */
public class PrintViaViewer {

    /**
     * Show the print dialog and print asynchrony.
     * @param view the current ReportView
     * @return the PrinterJobProgress
     */
    private static Progress printWithPrintDialog( ReportView view ) {
        return view.print( 1, -1, true );
    }

    /**
     * Print asynchrony directly without asking the user for any.
     * @param view the current ReportView
     * @return the PrinterJobProgress
     */
    private static Progress printWithoutPrintDialog( ReportView view ) {
        return view.print( 1, -1, false );
    }

    /**
     * Print asynchrony with custom attributes and show the print dialog of the viewer.
     * @param view the current ReportView
     * @return the PrinterJobProgress
     */
    private static Progress printWithCustomAttributes( ReportView view ) {
        HashPrintRequestAttributeSet printAttributes = view.getDefaultAttributeSet( 1, -1 );

        //set some optional pre set print attributes for the print dialog
        printAttributes.add( new Copies( 2 ) );

        PrinterJob printerJob = PrinterJob.getPrinterJob(); //default printer
        if( !SwingReportView.showPrintDialog( (Component)view, printerJob, printAttributes ) ) {
            return null; // user has cancel
        }

        //set some optional print attributes that the user should not changed
        printAttributes.add( new JobName( "My Print Job", null ) );

        // print asynchrony
        return view.print( printerJob, printAttributes );
    }

    /**
     * Main method of the sample.
     * @param args no parameters used
     * @throws IOException in case of IO error (e.g. port already in use)
     */
    public static void main( String[] args ) throws IOException {

        // The instance of the viewer in your application
        SwingReportViewer viewer = new SwingReportViewer();

        String reportLocation = "";
        try {
            reportLocation = PrintViaViewer.class.getResource( "../sample.rpt" ).toURI().toString();
        } catch( URISyntaxException e ) {
            // Nothing to do
        }

        // then initialize the render data connection.
        RenderData data = new EngineRenderData( "report=" + reportLocation );

        // you will most likely have a report server already, so you can use the URL: http://server:port/report.rpt
        // RenderData data = new URLRenderData( "http://server:port/?report=file:c:/report1.rpt" );

        // Create a not visible ReportView if you does not have one
        ReportView view = viewer.createReportView( data );
        viewer.addReportView( view );

        Progress progress = printWithPrintDialog( view );
        //Progress progress = printWithoutPrintDialog( view );
        //Progress progress = printWithCustomAttributes( view );

        if( progress != null ) {
            progress.waitUntilFinished(); // wait before we terminate the Java process
        }
        viewer.closeReportView( view );
    }
}
